module Global where
import Data.Map
import Data.Set as S
import System.IO
import Control.Concurrent.MVar
import Control.Concurrent
import Control.Monad
import System.Process
import System.Posix.Signals
import System.Posix.Process
import System.Posix.Types
import Text.Regex.Posix
import Random

sendCommand :: Worker -> MVar Bool -> ( State -> IO State ) -> IO ()
sendCommand worker response f = do
	putMVar worker $ \state -> catch ( do
		newState <- f state
		putMVar response True >> return newState ) $ const ( putMVar response False >> return state )
	result <- takeMVar response
	if result then return () else ioError $ userError ""

runWaiter :: Worker -> ProcessHandle -> Integer -> IO ()
runWaiter worker h songID = forkIO ( catch ( waitForProcess h >> newEmptyMVar >>= \response -> sendCommand worker response $ nextCond songID ) $ const $ return () ) >> return ()

runReader :: Handle -> IO ()
runReader handle = ( forkIO $ catch ( forever $ hGetChar handle >>= putChar ) $ const $ return () ) >> return ()

data State = State {
	playlist :: [ String ],
	history :: [ String ],
	act :: String,
	modes :: Set String,
	player :: Player,
	sWorker :: Worker,
	startNext :: Integer
	}

type Task = State -> IO State
type Worker = MVar Task

data PlayerState = Stopped | Paused | Playing deriving ( Eq )

class PlayerClass p where
	pauseP :: p -> IO p
	resumeP :: p -> IO p
	stopP :: p -> IO p
	playerStateP :: p -> PlayerState

data NullPlayer = NullPlayer
instance PlayerClass NullPlayer where
	pauseP = return
	resumeP = return
	stopP = return
	playerStateP = const Global.Stopped

data MPlayer = MPlayer {
	mpState :: PlayerState,
	mpPid :: ProcessHandle,
	mpPeer :: Handle
	}
instance PlayerClass MPlayer where
	pauseP pl = catch ( hPutChar ( mpPeer pl ) ' ' ) ( const $ return () ) >> ( return $ pl { mpState = if mpState pl == Playing then Paused else Playing } )
	resumeP = pauseP
	stopP pl = catch ( hPutChar ( mpPeer pl ) 'q' ) ( const $ return () ) >> ( return $ pl { mpState = Global.Stopped } )
	playerStateP = mpState
newMPlayer :: State -> String -> IO State
newMPlayer state file = do
	( inp, out, err, pid ) <- runInteractiveProcess "/usr/bin/mplayer" ( ( if "novideo" `S.member` modes state then [ "-vo", "null" ] else [] ) ++ [ file ] ) Nothing Nothing
	hSetBuffering inp NoBuffering
	runReader out
	runReader err
	runWaiter ( sWorker state ) pid $ startNext state
	return $ state { player = Player $ MPlayer { mpState = Playing, mpPid = pid, mpPeer = inp } }

data MyPlay = MyPlay {
	myState :: PlayerState,
	myPid :: ProcessID
	}
instance PlayerClass MyPlay where
	pauseP = doSignal softwareStop Paused
	resumeP = doSignal continueProcess Playing
	stopP = doSignal softwareTermination Global.Stopped
	playerStateP = myState
doSignal :: Signal -> PlayerState -> MyPlay -> IO MyPlay
doSignal signal state pl = ( signalProcess signal $ myPid pl ) >> ( return $ pl { myState = state } )
newMyPlay :: State -> String -> IO State
newMyPlay state file = forkProcess ( executeFile "myplay" True [ file ] Nothing ) >>= \pid ->
	forkIO ( getProcessStatus True False pid >> newEmptyMVar >>= \response -> sendCommand ( sWorker state ) response $ nextCond $ startNext state ) >>
	( return $ state { player = Player $ MyPlay { myState = Playing, myPid = pid } } )

data Player = forall p. ( PlayerClass p ) => Player p

play :: MVar Task -> String -> IO Player
play = const $ const $ return $ Player NullPlayer

stop :: State -> IO State
stop state = case player state of Player p -> stopP p >> ( return $ state { player = Player NullPlayer, startNext = startNext state + 1 } )

pause :: State -> IO State
pause state = case player state of Player p -> pauseP p >>= \pl -> return $ state { player = Player pl }

resume :: State -> IO State
resume state = case player state of Player p -> resumeP p >>= \pl -> return $ state { player = Player pl }

nextCond :: Integer -> State -> IO State
nextCond id state = if id == startNext state then Global.next $ state { player = Player NullPlayer } else return state

changeSong :: ( State -> Maybe ( State, String ) ) -> State -> IO State
changeSong f state = case f state of
	Nothing -> return state
	Just ( ms, fn ) -> catch ( stop ms >>= playFile fn >>= \state -> return state ) ( const $ return state ) >>= \state -> return state

next :: State -> IO State
next state = ( randomIO :: IO Int ) >>= ( \rand -> changeSong ( chooseSong rand ) state )
	where
		chooseSong :: Int -> State -> Maybe ( State, String )
		chooseSong rand state
			| Prelude.null $ playlist state = Nothing
			| "random" `S.member` ( modes state ) = let
					pl = playlist state
					offset = rand `mod` length pl
					start = take offset pl
					end = drop offset pl
					fl = head end
				in rep $ ( state { history = fl : history state, playlist = start ++ tail end }, fl )
			| otherwise = rep $ ( state { history = ( head $ playlist state ) : history state, playlist = tail $ playlist state }, head $ playlist state )
		rep :: ( State, String ) -> Maybe ( State, String )
		rep ( state, fn ) = Just ( checkHSize $ state { playlist = playlist state ++ [ fn ] }, fn )
		checkHSize :: State -> State
		checkHSize state
			| "repeat" `S.member` ( modes state ) = state { history = take 100 $ history state }
			| otherwise = state

prev :: State -> IO State
prev = changeSong ( \state -> if Prelude.null $ history state then Nothing else Just ( state { history = tail $ history state, playlist = if "repeat" `S.member` ( modes state ) then ( head $ history state ) : playlist state else playlist state }, head $ history state ) )

playerState :: State -> PlayerState
playerState state = case player state of Player p -> playerStateP p

playFile :: String -> State -> IO State
playFile file state = ( putStrLn $ "Playing file " ++ file ) >> let newState = state { startNext = startNext state + 1, act = file } in
	if file =~ "\\.(ogg|waw|mp3)" then
		newMyPlay newState file
	else
		newMPlayer newState file

playPause :: State -> IO State
playPause state = case playerState state of
	Global.Stopped -> Global.next state
	Paused -> resume state
	Playing -> pause state
