/*
   Sticky
   Copyright (C) 2006 Michal Vaner (michal.vaner@kdemail.net)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   he Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "event.h"
#include <QSettings>
#include <QTimer>

void Event::setText( const QString &text ) { mText = text; checkState(); }
void Event::setName( const QString &name ) { mName = name; checkState(); }
void Event::setDate( bool hasDate, const QDate &date ) {
	mHasDate = hasDate;
	mDate.setDate( date );
	checkState();
}
void Event::setTime( bool hasTime, const QTime &time ) {
	mHasTime = hasTime;
	mDate.setTime( time );
	checkState();
}
void Event::setTodo( bool todo ) { mTodo = todo; checkState(); }
bool Event::isLate() const {
	if( !hasDate() )
		return false;
	else if( !hasTime() ) {
		QDateTime date = mDate;
		date.setTime( QTime() );
		QDateTime today = QDateTime::currentDateTime();
		today.setTime( QTime() );
		today = today.addSecs( 12 * 60 * 60 );
		return( today > date );
	} else {
		return( QDateTime::currentDateTime().addSecs( 12 * 60 * 60 ) > mDate );
	}
}

bool Event::isReallyLate() const {
	if( !hasDate() )
		return false;
	else if( !hasTime() ) {
		QDateTime date = mDate;
		date.setTime( QTime() );
		QDateTime today = QDateTime::currentDateTime();
		today.setTime( QTime() );
		return( today > date );
	} else {
		return( QDateTime::currentDateTime() > mDate );
	}
}

bool Event::isNear() const {
	if( isReallyLate() )
		return false;
	else if( !hasDate() )
		return false;
	else if( !hasTime() ) {
		QDateTime date = mDate;
		date.setTime( QTime() );
		QDateTime today = QDateTime::currentDateTime();
		today.setTime( QTime() );
		today = today.addDays( 7 );
		return( today > date );
	} else {
		QDateTime date = mDate;
		return( QDateTime::currentDateTime().addDays( 7 ) > mDate );
	}
}

static QTimer *timer = 0L;

Event::Event() :
	mText(),
	mName( "noname" ),
	mDate( QDateTime::currentDateTime() ),
	mHasTime( false ),
	mHasDate( true ),
	mTodo( false )
{
	checkState();
	if( !timer ) {
		timer = new QTimer;
		timer->start( 60 * 1000 );
	}
	connect( timer, SIGNAL(timeout()), SLOT(checkState()) );
}

Event::Event( QSettings &s ) :
	mText( s.value( "text" ).toString() ),
	mName( s.value( "name" ).toString() ),
	mDate( s.value( "date" ).toDateTime() ),
	mHasTime( s.value( "hasTime" ).toBool() ),
	mHasDate( s.value( "hasDate" ).toBool() ),
	mTodo( s.value( "todo" ).toBool() )
{
	checkState();
	if( !timer ) {
		timer = new QTimer;
		timer->start( 60 * 1000 );
	}
	connect( timer, SIGNAL(timeout()), SLOT(checkState()) );
}

void Event::save( QSettings &s ) {
	s.setValue( "text", mText );
	s.setValue( "name", mName );
	s.setValue( "date", mDate );
	s.setValue( "hasTime", mHasTime );
	s.setValue( "hasDate", mHasDate );
	s.setValue( "todo", mTodo );
}

void Event::checkState() {
	bool em = ( ( lastTodo != isTodo() ) || ( lastNear != isNear() ) || ( lastLate != isLate() ) );
	lastTodo = isTodo();
	lastNear = isNear();
	lastLate = isLate();
	if( em )
		emit stateChanged();
}

QList< Event * > Events::todo() {
	QList< Event * > all = events(), res;
	foreach( Event *e, all )
		if( e->isTodo() )
			res << e;
	return res;
}

QList< Event * > Events::late() {
	QList< Event * > all = events(), res;
	foreach( Event *e, all )
		if( e->isLate() )
			res << e;
	return res;
}

QList< Event * > Events::near() {
	QList< Event * > all = events(), res;
	foreach( Event *e, all )
		if( e->isNear() )
			res << e;
	return res;
}

QList< Event * > Events::events() { return mEvents; }

void Events::addEvent( Event *e ) {
	mEvents << e;
	emit update();
	connect( e, SIGNAL(stateChanged()), SIGNAL(update()) );
}

void Events::delEvent( Event *e ) {
	mEvents.removeAll( e );
	delete e;
	emit update();
}

Events *events = 0L;

void Events::load() {
	QSettings set( "vorner", "sticky" );
	int size = set.beginReadArray( "notes" );
	for( int i = 0; i < size; ++i ) {
		set.setArrayIndex( i );
		addEvent( new Event( set ) );
	}
}

void Events::save() {
	QSettings set( "vorner", "sticky" );
	set.beginWriteArray( "notes" );
	int i = 0;
	foreach( Event *e, mEvents ) {
		set.setArrayIndex( i++ );
		e->save( set );
	}
	set.endArray();
}

#include "moc_event.cpp"
