/*
   Sticky
   Copyright (C) 2006 Michal Vaner (michal.vaner@kdemail.net)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   he Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the

   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtGlobal>
#if QT_VERSION <= 0x040200
#error "Too old version of Qt, need at last 4.2"
#endif

#include "show.h"
#include "event.h"
#include <QSettings>

Show *show = 0L;

Show::Show() :
	selected( 0L ),
	locked( false )
{
	setupUi( this );
	connect( allButt, SIGNAL(clicked()), SLOT(update()) );
	connect( nearButt, SIGNAL(clicked()), SLOT(update()) );
	connect( lostButt, SIGNAL(clicked()), SLOT(update()) );
	connect( todoButt, SIGNAL(clicked()), SLOT(update()) );
	connect( events, SIGNAL(update()), SLOT(update()) );
	itemList->setSortingEnabled( true );
}

class MyItem : public QListWidgetItem {
	public:
		Event *event;
		MyItem( const QString &text, Event *ev, QListWidget *parent = NULL ) :
			QListWidgetItem( text, parent ),
			event( ev )
		{}
		virtual bool operator<( const QListWidgetItem &other ) const {
			return( event->dateTime() < ( ( const MyItem & ) other ).event->dateTime() );
		}
};

void Show::on_remButt_clicked() {
	Event *e = ( (MyItem *) itemList->currentItem() )->event;
	selected = 0L;
	events->delEvent( e );
	events->save();
}

void Show::update() {
	QList< Event * > evs;
	if( allButt->isChecked() )
		evs = events->events();
	else if( nearButt->isChecked() )
		evs = events->near();
	else if( lostButt->isChecked() )
		evs = events->late();
	else
		evs = events->todo();
	itemList->clear();
	foreach( Event *e, evs ) {
		MyItem *i = new MyItem( e->name(), e );
		itemList->addItem( i );
		if( e->isLate() )
			i->setTextColor( Qt::red );
		else if( e->isNear() )
			i->setTextColor( QColor( 200, 200, 0 ) );
		else if( e->isTodo() )
			i->setTextColor( Qt::blue );
		if( e == selected ) {
			itemList->setCurrentItem( i );
		}
	}
}

void Show::on_addButt_clicked() {
	allButt->click();
	Event *ev = new Event();
	selected = ev;
	events->addEvent( ev );
	events->save();
}

void Show::on_itemList_currentItemChanged( QListWidgetItem *it, QListWidgetItem * ) {
	if( locked )
		return;
	locked = true;
	remButt->setEnabled( it );
//	frame->setEnabled( it );
	nameEdit->setEnabled( it );
	textEdit->setEnabled( it );
	todoCheck->setEnabled( it );
	dateCheck->setChecked( false );
	timeCheck->setChecked( false );
	dateCheck->setEnabled( it );
	if( it ) {
		selected = ( (MyItem *) it )->event;
		nameEdit->setText( selected->name() );
		bool hasDate = selected->hasDate();
		bool hasTime = selected->hasTime();
		dateCheck->setChecked( hasDate );
		dateEdit->setDate( selected->date() );
		timeCheck->setChecked( hasTime );
		timeEdit->setTime( selected->time() );
		todoCheck->setChecked( selected->isTodo() );
		textEdit->setPlainText( selected->text() );
		nearState->setEnabled( selected->isNear() );
		lostState->setEnabled( selected->isLate() );
		todoState->setEnabled( selected->isTodo() );
	}
	locked = false;
}

void Show::on_nameEdit_textEdited( const QString &text ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setName( text );
	itemList->currentItem()->setText( text );
	events->save();
}

void Show::on_dateCheck_toggled( bool val ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setDate( val, dateEdit->date() );
	events->save();
}

void Show::on_dateEdit_dateChanged( const QDate &date ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setDate( dateCheck->isChecked(), date );
	events->save();
}

void Show::on_textEdit_textChanged() {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setText( textEdit->toPlainText() );
	events->save();
}

void Show::on_timeCheck_toggled( bool val ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setTime( val, timeEdit->time() );
	events->save();
}

void Show::on_timeEdit_timeChanged( const QTime &time ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setTime( timeCheck->isChecked(), time );
	events->save();
}

void Show::on_todoCheck_toggled( bool val ) {
	if( locked )
		return;
	if( !selected )
		return;
	selected->setTodo( val );
	events->save();
}

void Show::closeEvent( QCloseEvent *ev ) {
	QSettings().sync();
	QMainWindow::closeEvent( ev );
}

#include "moc_show.cpp"
